'''
API which handles logic behind the database server.
This API is designed to handle database queries and return results in a structured format.

Run for testing with:
CUDA_VISIBLE_DEVICES=0 python /scratch/dhoward/Chatbot/sql_server.py
To run this command locally, hba.conf files will need to be edited to specify single gpu usage. 

For server running:
sudo supervisorctl start / stop / status myadvisor-chatbot-sql-server
'''

from fastapi import FastAPI, HTTPException
from pydantic import BaseModel
from sqlbot import QueryWithExecution, check_database_status, generate_and_execute_query

app = FastAPI(
    title="Database testing API",
    description="API for testing how data is extracted",
    version="2.0.1"
)

@app.get("/health")
async def health_check():
    """
    Health check endpoint to verify if the database is connected and the model is loaded.
    """
    try:
        status = check_database_status()
        return {
            "status": "Why is there an error?",
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Health check error: {str(e)}")

@app.get("/status")
async def status_check():
    """
    Status check endpoint to verify if the database is connected and the model is loaded.
    """
    try:
        status = check_database_status()
        return {
            "status": "healthy",
        }
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"Status check error: {str(e)}")

@app.post("/query")
async def query_database(request: QueryWithExecution):
    """
    Endpoint to handle database queries.
    This endpoint generates and executes a query based on the provided question.
    Returns the query result and model information.
    """
    try:
        query = QueryWithExecution(question=request.question, max_rows=request.max_rows, debug=request.debug, indices=request.indices if hasattr(request, 'indices') else None)
        result = generate_and_execute_query(query)
        return {
            "executed": "Executed query successfully",
            "user_question": request.question,
            "query": result.sql_query,
            "results": result.results,
            "execution_time": result.execution_time_ms,
        }
    except Exception as e:
        raise HTTPException(
            status_code=500, 
            detail=f"Query error: {str(e)}")

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8002)
